<?php

namespace App\Http\Controllers;

use Symfony\Component\HttpFoundation\StreamedResponse;
use App\Models\Auditionee;
use App\Models\VenueDate;
use App\Models\Course;
use App\Models\Project;
use App\Models\Role;
use App\Enums\Country;
use Illuminate\Support\Facades\URL;

class ExportCsvController extends Controller
{
    public function export($model, $ids)
    {
        // Define the headers
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="export.csv"',
        ];

        // Return the streamed response
        return new StreamedResponse(function () use ($model, $ids) {
            $handle = fopen('php://output', 'w');

            // Determine which model to use
            switch ($model) {
                case 'auditionees':
                    $data = $this->getAuditionees($ids);
                    break;
                case 'course':
                    $data = $this->getAttendees($ids);
                    break;
                case 'project':
                    $data = $this->getRoles($ids);
                    break;
                case 'cpp':
                    $data = $this->getCpp($ids); // we need a modifier on this
                    break;
                default:
                    fclose($handle);
                    abort(404, 'Model not found');
            }

            // Add headers to the CSV
            fputcsv($handle, $data[0]);

            // Add each user's data to the CSV
            foreach ($data[1] as $datarow) {
                fputcsv($handle, $datarow);
            }

            fclose($handle);
        }, 200, $headers);
    }

    public function getCpp($id)
    {
        // Retrieve auditionees with specific criteria
        $auditionees = Auditionee::where('id', '>', 0)
            ->where('category', '=', 'cpp')
            ->get();  // Make sure to fetch the results

        $data = [];

        // Define the fields to be exported
        $fieldsArray = [
            'First Name',
            'Last Name',
            'Role',
            'Gender',
            'DOB',
            'Age',
            'Main Contact Email', 
            'Secondary Email',
            'Town / City',
            'Have you taken part in NYMT before?', 
            'Allergies',
            'Epipen user',
            'Inhaler user',
            'Access',
            'Availability issues',
            'Course Link'
        ];

        // Set the headers for the CSV
        $data[0] = $fieldsArray;

        // Loop through each auditionee and prepare data for CSV
        foreach ($auditionees as $auditionee) {
            
            
            $detailsUrl = 
            URL::temporarySignedRoute(
            'ycm.course', now()->addDays(180), [
                    'id' => $auditionee->id,
                ]
            );
            
            $data[1][] = [
                ucwords($auditionee->firstname),
                ucwords($auditionee->lastname),
                $auditionee->ycm_type(),
                $auditionee->gender,
                $auditionee->dob,
                $auditionee->age(),
                $auditionee->email,
                $auditionee->secondary_email, // secondary_email
                $auditionee->town, // town_city
                $auditionee->nymt_experience."|".$auditionee->nymt_experience_yes, // have_you_taken_part_in_nymt_before
                $auditionee->pastoral?->allergies, // allergies
                $auditionee->pastoral?->epipen, // epipen_user
                $auditionee->pastoral?->inhaler, // inhaler_user
                $auditionee->pastoral?->barriers, // barriers
                $auditionee->unavailability, // availability_issues
                $detailsUrl // link
            ];
        }

        return $data;
    }
    

    public function getAuditionees($id)
    {
        $venuedate = VenueDate::find($id);
        // Retrieve auditionees with specific criteria
        $auditionees = Auditionee::where('id', '>', 0)
            ->whereRelation('auditions', 'venue_date_id', '=', $id)
            ->get();  // Make sure to fetch the results

        $data = [];

        // Define the fields to be exported
        $fieldsArray = [
            'First Name',
            'Pronunciation',
            'Last Name',
            'Category',
            'Audition Name',
            'Audition Venue Date',
            'Gender',
            'DOB',
            'Age',
            'Main Contact Email', 
            'Secondary Email',
            'Town / City',
            'Voice',
            'Musical Instruments',
            'Have you taken part in NYMT before?', 
            'Allergies',
            'Epipen user',
            'Inhaler user',
            'Availability issues',
            'Access',
            'Ethnicity',
            'Acting Score', 
            'Acting Comment', 
            'Singing Score', 
            'Singing Comment', 
            'Dancing Score', 
            'Dancing Comment',
            'Outcome',
            'Potential roles',
            'Course Link'
        ];

        // Set the headers for the CSV
        $data[0] = $fieldsArray;

        // Loop through each auditionee and prepare data for CSV
        foreach ($auditionees as $auditionee) {
            
            
            $detailsUrl = 
            URL::temporarySignedRoute(
            'ycm.course', now()->addDays(180), [
                    'id' => $auditionee->id,
                ]
            );
            
            $data[1][] = [
                ucwords($auditionee->firstname),
                $auditionee->pastoral?->comment, // name pronunciation
                ucwords($auditionee->lastname),
                $auditionee->ycm_type(),
                $venuedate->name,
                date('Y-m-d' , strtotime($venuedate->start_date)),
                $auditionee->gender,
                $auditionee->dob,
                $auditionee->age(),
                $auditionee->email,
                $auditionee->secondary_email, // secondary_email
                $auditionee->town, // town_city
                $auditionee->singing_experience, // voice
                $auditionee->musical_instruments, // musical_instruments
                $auditionee->nymt_experience."|".$auditionee->nymt_experience_yes, // have_you_taken_part_in_nymt_before
                $auditionee->pastoral?->allergies, // allergies
                $auditionee->pastoral?->epipen, // epipen_user
                $auditionee->pastoral?->inhaler, // inhaler_user
                $auditionee->unavailability, // availability_issues
                $auditionee->pastoral?->barriers, // barriers
                $auditionee->ethnicity, // ethnicity
                $auditionee->audition->score?->acting_score,
                $auditionee->audition->score?->acting_comment,
                $auditionee->audition->score?->singing_score,
                $auditionee->audition->score?->singing_comment,
                $auditionee->audition->score?->dancing_score,
                $auditionee->audition->score?->dancing_comment, // $audtionee->auditions->acting_score
                $auditionee->audition->score?->outcome,  // outcome
                $auditionee->audition->score?->role_info, // potential_roles
                $detailsUrl // link
            ];
        }

        return $data;
    }
    
    
    public function getAttendees($id) {
        

        $project_id = 0;
        $course = Course::find($id);
        $auditionees = Auditionee::where('id', '>', 0)
                    ->whereRelation('courses', 'course_id', '=', $id)
                    ->get();  // Make sure to fetch the results
        $projectname  = $course->name. ":".$course->location.":".date('Y-m-d' , strtotime($course->start_date)); 

        
        // Retrieve auditionees with specific criteria

        $data = [];

        // Define the fields to be exported
        $fieldsArray = [
            'First Name', 'Pronunciation', 'Last Name', 'Category', 'Project', 'Role', 'Main Contact Email', 'Secondary Email', 'Phone',
            'Gender', 'Sex at birth', 'DOB', 'Age', 'Formal Name', 
            'Town/City', 'Country', 'Epipen user', 'Inhaler user', 'Disability', 
            'Allergy Information', 
            'Medical information: Please tell us if you have recently experienced', 
            'Medical conditions: more info', 
            'Medical conditions: other info', 
            'Dietary requirements', 
            'Homesickness',
            'Course Link'
        ];

        // Set the headers for the CSV
        $data[0] = $fieldsArray;

        // Loop through each auditionee and prepare data for CSV
        foreach ($auditionees as $auditionee) {
            $castrole ="N/A";
            
            
            
            $detailsUrl = 
            URL::temporarySignedRoute(
            'ycm.course', now()->addDays(180), [
                    'id' => $auditionee->id,
                ]
            );
            
            
            $roles = $auditionee->roles;
            if ($roles) {
                foreach ($roles as $role){
                    if ($role->role->project_id == $project_id) { // can they have more than one role in same project?
                        $castrole = $role->role->name;
                    }
                }
            } 
            
            $data[1][] = [
                ucwords($auditionee->firstname), // First Name
                $auditionee->pastoral?->comment, // name pronunciation
                ucwords($auditionee->lastname), // Last Name
                $auditionee->category, // Category
                $projectname, // Project
                $castrole, // Role
                $auditionee->email, // Main Contact Email
                $auditionee->secondary_email, // Secondary Email
                $auditionee->tel ? '="' . $auditionee->tel . '"'   // → ="07987654321"
                : '', // Phone
                $auditionee->gender, // Gender
                $auditionee->birth_gender, // Sex at birth
                $auditionee->dob, // DOB
                $auditionee->age(), // Age
                $auditionee->formalname, // Formal Name
                $auditionee->town, // Town/City
                Country::from($auditionee->country->value)->name, // Country
                $auditionee->pastoral?->epipen, // Epipen user
                $auditionee->pastoral?->inhaler, // Inhaler user
                $auditionee->pastoral?->disability, // Disability
                $auditionee->pastoral?->allergies,  // Allergy Information
                $auditionee->casting?->medical_info, // Medical information: Please tell us if you have recently experienced
                $auditionee->casting?->medical_more_info, // Medical conditions: more info
                $auditionee->casting?->medical_info_other,// Medical conditions: other info
                $auditionee->casting?->dietry, // Dietary requirements
                $auditionee->casting?->homesickness, // Homesickness
                $detailsUrl // link
            ];
        }

        return $data;
    }
    
    
    public function getRoles($id) {

        $project = Project::find($id);
        $project_id = $id;

        $roles = Role::where('project_id','=',$project_id)->orderBy('position', 'asc')->get();
        
        // Retrieve auditionees with specific criteria

        $data = [];

        // Define the fields to be exported
        $fieldsArray = [
            'First Name', 
            'Pronunciation',
            'Last Name', 
            'Category', 
            'Project', 
            'Role', 
            'Role Status', 
            'Main Contact Email', 
            'Secondary Email',
            'Phone',
            'Gender', 
            'Sex at birth', 
            'DOB', 
            'Age', 
            'Formal Name', 
            'Town/City', 
            'Country', 
            'Epipen user', 'Inhaler user',
            'Disability', 
            'Allergy Information',
            'Accessibility',
            'Medical information: Please tell us if you have recently experienced',
            'Medical conditions: more info',
            'Medical conditions: other info',
            'Dietary requirements',
            'Homesickness',
            'Link'
        ];

        // Set the headers for the CSV
        $data[0] = $fieldsArray;

        // Loop through each auditionee and prepare data for CSV
        foreach ($roles as $role) {
            $country = "";
            if ($role->audition_role?->auditionee->country){
                $country = Country::from($role->audition_role?->auditionee->country->value)->name;
            }
            
            $detailsUrl = "no link";
            $roleStatus = "No YCM Chosen";
            
            if ($role->audition_role){
                
                $detailsUrl = 
                URL::temporarySignedRoute(
                'ycm.casting', now()->addDays(5), [
                        'projectid'  => $project_id,
                        'id' => $role->audition_role?->auditionee->id,
                    ]
                );
                
                switch($role->audition_role->status) {
                    case 2:
                        $roleStatus = "Accepted";
                    break;
                    case 0:
                        $roleStatus = "Declined";
                    break;
                    default:
                        $roleStatus = "Offered";
                    break;
                }
            }
            
            $data[1][] = [
                ucwords($role->audition_role?->auditionee->firstname), // First Name
                $role->audition_role?->auditionee->pastoral?->comment, // name pronunciation
                ucwords($role->audition_role?->auditionee->lastname), // Last Name
                $role->audition_role?->auditionee->category, // Category
                $project->name, // Project
                $role->name, // Role
                $roleStatus, // Role
                $role->audition_role?->auditionee->email, // Main Contact Email
                $role->audition_role?->auditionee->secondary_email, // Secondary Email
                // Phone – force Excel to treat it as text
                $role->audition_role?->auditionee?->tel
                ? '="' . $role->audition_role->auditionee->tel . '"'   // → ="07987654321"
                : '',
                $role->audition_role?->auditionee->gender, // Gender
                $role->audition_role?->auditionee->birth_gender, // Sex at birth
                $role->audition_role?->auditionee->dob, // DOB
                $role->audition_role?->auditionee->age(), // Age
                $role->audition_role?->auditionee->formalname, // Formal Name
                $role->audition_role?->auditionee->town, // Town/City
                $country, // Country
                $role->audition_role?->auditionee->pastoral?->epipen, // Epipen user
                $role->audition_role?->auditionee->pastoral?->inhaler, // Inhaler user
                $role->audition_role?->auditionee->pastoral?->disability, // Disability
                $role->audition_role?->auditionee->pastoral?->allergies,  // Allergy Information
                $role->audition_role?->auditionee->pastoral?->barriers,  // Accessibility
                $role->audition_role?->auditionee->casting?->medical_info, // Medical information: Please tell us if you have recently experienced
                $role->audition_role?->auditionee->casting?->medical_more_info, // Medical conditions: more info
                $role->audition_role?->auditionee->casting?->medical_info_other,// Medical conditions: other info
                $role->audition_role?->auditionee->casting?->dietry, // Dietary requirements
                $role->audition_role?->auditionee->casting?->homesickness, // Homesickness
                $detailsUrl // link
            ];
        }

        return $data;
    }
    
}
