<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

use Illuminate\Foundation\Auth\User as Authenticatable;
use App\Enums\Country;
use App\Models\CommentCategory;
use App\Models\AuditioneeParent;


class Auditionee extends Authenticatable
{
    
    use HasFactory;


    protected $appends = ['age'];

    protected $guarded = [];
    
    
    protected $casts = [
        'country' => Country::class,
        'dancing_experience' => 'array', // Replace 'your_column' with your actual column name
    ];
    
    

    
    
    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];
    
    protected static function booted()
    {
        static::saving(function ($auditionee) {
            if (!$auditionee->auditionee_parent_id && $auditionee->email) {
                $firstname = $auditionee->firstname ?? '(Unknown)';
                $lastname = $auditionee->lastname ?? '(Unknown)';
                
                $parent = AuditioneeParent::firstOrCreate(
                    ['email' => $auditionee->email],
                    [
                        'firstname' => $auditionee->firstname,
                        'lastname' => $auditionee->lastname,
                    ]
                );
    
                $auditionee->auditionee_parent_id = $parent->id;
            }
        });
    }

    public function guardian(): BelongsTo
    {
        return $this->belongsTo(AuditioneeParent::class, 'auditionee_parent_id');
    }
    
    public function roles(){
       return $this->hasMany(AuditioneeRole::class);
    }
    public function casting(){
       return $this->hasOne(AuditioneeCasting::class);
    }
    public function comments(){
       return $this->hasMany(AuditioneeComment::class);
    }
    public function pastoral(){
       return $this->hasOne(AuditioneePastoral::class);
    }
    public function recalls(){
       return $this->hasMany(AuditioneeRecall::class);
    }
    public function transactions(){
       return $this->hasMany(AuditioneeTransaction::class);
    }
    
    // need to check on plural
    public function courses(){
       return $this->hasMany(Attendee::class);
    }
    public function course(){
       return $this->hasOne(Attendee::class)->latestOfMany();
    }
    // need to retire the plural
    public function auditions(){
       return $this->hasMany(Audition::class);
    }
    public function audition(){
        return $this->hasOne(Audition::class)->latestOfMany();
    }
    public function media(){
       return $this->hasMany(AuditioneeMedia::class);
    }
    
    
    public function group(){
        $auditioneesTotal = count($this->auditions());
       return count($this->auditions());
    }
    
    
    public function scores($type = 'acting'){
        $score = 0;
        $i = 0;
        foreach($this->comments as $comment){
            if ($comment->commentgroup == $type){
            $score = $score + $comment->score;
            $i++;
            }
        }
        if ($i>0){
            $average = $score / ($i);
            return number_format($average,2);
        } else {
            return $i;
        }
        
    }
    
    
    public function scoresDone($type = 'acting'){
        $i = 0;
        foreach($this->comments as $comment){
            if ($comment->commentgroup == $type){
            $i++;
            }
        }
        
        $query = CommentCategory::where('id','>',0);
        $query->where('type','=',$type);
        
        $catnum = $query->count();
        
        return $i." of ".$catnum;
        
    }
    
    
    public function getAgeAttribute()
    {
        return Carbon::parse($this->attributes['dob'])->age;
    }
    
    public function ageAtDate($futureDate)
    {
        $dob = Carbon::parse($this->attributes['dob']);
        $future = Carbon::parse($futureDate);
        return floor($dob->diffInYears($future));
    }
    
    public function age(){
        
        $dob = $this->dob;
        if ($dob) {
            $year = (date('Y') - date('Y',strtotime($dob)));
        } else {
            $year = date('Y');
        }
        return $year;
    }
    
    
    public function paid(){
        $result = false;
        if ($this->audition){
            $result .= "Audition: ".$this->audition->paid()."<br />";
        }
        if ($this->course){
            $result .= "Course: ".$this->course->paid()."<br />";
        }
        if ($this->category == 'cpp'){
            $paidCPP = false;
            foreach ($this->transactions as $transaction){
                if ($transaction->type == "CPP" && $transaction->status == "success"){
                    $paidCPP = true;
                    $result .= "CPP: Yes - £".$transaction->amount;
                }
            }
            if (!$paidCPP){
                    $result .= "CPP: Not Yet";
            }
            
        }
        
        if ($result) {
            return $result;
        }
        
        return "No";
        
    }
    
    public function ycm_type(){
        
        switch($this->category) {
            case 'actor':
                $result = "Actor";
            break;
            case 'cpp':
                $result = $this->nymt_role;
            break;
            default:
                $result = ucwords(str_replace('_',' ', $this->category));
            break;
            }
            
        return $result;
    }
    
    
    public function tags(){
        // https://tailwindcss.com/docs/customizing-colors
        $tags = array();
        if ($this->licensed > 0){
            $tags['lime-500'] = 'Licensed Performer';
        }
        if ($this->category > 0){
            
            switch($this->category) {
                case 'actor':
                    $result = "Actor";
                break;
                case 'cpp':
                    $result = "Creative & Production Pathway";
                break;
                default:
                    $result = ucwords(str_replace('_',' ', $this->category));
                break;
            }
            $tags['blue-400'] = $result;
        }
        if ($this->age() > 18){
            //$tags['slate-500'] = 'Adult';
        } elseif ($this->age() >= 16 && $this->age() < 19){
           // $tags['slate-500'] = 'Under 18';
        } else {
            $tags['teal-500'] = 'Under 16';
        }
        
        
        if ($this->film_permission>1){
            $tags['slate-500'] = 'Media';
        }
        
        if ($this->pastoral){
            if ($this->pastoral->epipen > 0){
                $tags['red-400'] = 'Epipen User';
            }
    
            if ($this->pastoral->inhaler > 0){
                $tags['amber-400'] = 'Inhaler User';
            }
            
            if (strlen($this->pastoral->barriers)>0){
                $tags['amber-500'] = 'Access';
            }
        }
        
        // if ($this->casting && $this->casting->offsite==0){
        //    $tags['slate-500'] = 'Accompanied';
        // }
        
        if ($this->casting && $this->casting->offsite == 0 && strlen($this->casting->accept)>0) {
            $tags['slate-500'] = 'Accompanied';
        }
        
        
        // Under/over 18 on Day One of the project 
        // Recall status 
        //No Permission for Leaving Site
        //No permission for filming / photography

        return $tags;
    }

    
}